<?php

namespace Mnv\Admin\Controllers;

use Mnv\Core\Config;
use Mnv\Core\Database\Throwable\IntegrityConstraintViolationException;
use Mnv\Core\Design;
use Mnv\Core\UserInfo;
use Mnv\Core\Managers;

use Mnv\Core\Managers\AdminAuth;
use Mnv\Core\Managers\Errors\AuthError;
use Mnv\Core\Managers\Errors\UsernameRequiredError;
use Mnv\Core\Managers\Exceptions\UserRoleExistsException;
use Mnv\Core\Managers\Exceptions\InvalidPasswordException;
use Mnv\Core\Managers\Exceptions\TooManyRequestsException;
use Mnv\Core\Managers\Exceptions\UnknownUsernameException;
use Mnv\Core\Managers\Exceptions\AttemptCancelledException;
use Mnv\Core\Managers\Exceptions\EmailNotVerifiedException;
use Mnv\Core\Managers\Exceptions\ValidatePasswordException;
use Mnv\Models\Users\UserGroups;

/**
 * Class AuthAdmin
 * @package Mnv\Admin\Controllers
 */
class AuthAdmin
{
    private AdminAuth $authAdmin;
    private $smarty;

    private $login;
    private int $limit_error = 10;

    private ?int $rememberDuration;

    public array $errors = array();
    public array $messages = array();

    /**
     * Login constructor.
     */
    public function __construct()
    {

        try {
            $this->authAdmin = new AdminAuth(UserInfo::get_ip(), true, 60 * 5);
            $this->smarty    = Design::init()->design();
            loadLanguage(Config::init()->get('admin_language'));

            $action      = getRequestVar('action');
            $this->login = getRequestVar('login','',  true);

            if (Managers::init()->getUserBanned()) {
                $this->numberAttempts($this->authAdmin->getBanned());
            }

            if (isset($this->login['remember']) && (int)$this->login['remember'] === 1) {
                // оставаться в системе в течение одного года
                $this->rememberDuration = (int) (60 * 60 * 24 * 365.25);
            } else {
                // не оставаться в системе после окончания сеанса
                $this->rememberDuration = null;
            }

            if (!empty($action)) {
                if ($action === 'login') {
                    $this->adminLogin($this->authAdmin);
                } else if ($action === 'logout') {
                    $this->logout($this->authAdmin);
                }
            }

            $this->smarty->assign('login', $this->login);

            if (!empty($this->errors)) {
                $this->smarty->assign('errors', $this->errors);
            }
            if (!empty($this->messages)) {
                $this->smarty->assign('messages', $this->messages);
            }

        } catch (IntegrityConstraintViolationException | AuthError $e) {
            $this->smarty->assign('errors', $e->getMessage());
        }
    }


    /**
     * @param AdminAuth $authAdmin
     */
    public function adminLogin(AdminAuth $authAdmin): void
    {
        try {
            $this->login['loginName'] = htmlspecialchars($this->login['loginName']);
            $this->login['password']  = htmlspecialchars($this->login['password']);

            $authAdmin->loginAdminWithUsername($this->login['loginName'], $this->login['password'], $this->rememberDuration);

            $this->redirectRoleAdmin($authAdmin);

        }  catch (UsernameRequiredError $e) {
            $this->errors[] = lang('login:errors:5'); //lang('login:errors:0');
        } catch (UnknownUsernameException|InvalidPasswordException $e) {
            if (Managers::init()->getUserBanned()) {
                $authAdmin->setBanned();
            }
            $this->errors[] = 'Incorrect username or password.'; // Вы ввели неверный логин или пароль. Пожалуйста, попробуйте еще раз.
        }  catch (ValidatePasswordException $e) {
            $this->errors[] = lang('login:errors:6');
        } catch (EmailNotVerifiedException $e) {
            $this->errors[] = 'Email не подтвержден';
        } catch (UserRoleExistsException $e) {
            if (Managers::init()->getUserBanned()) {
                $authAdmin->setBanned();
            }
            $this->errors[] = 'Нет доступа для этого аккаунта';
        } catch (TooManyRequestsException $e) {
            if (Managers::init()->getUserBanned()) {
                Managers::init()->setUserBanned();
            }
            $this->errors[] = 'Слишком много запросов';
        } catch (AuthError | AttemptCancelledException $e) {
            $this->errors[] = $e->getMessage();
        }
    }

    /**
     * @param AdminAuth $authAdmin
     */
    private function redirectRoleAdmin(AdminAuth $authAdmin): void
    {
        if ($authAdmin->hasAnyRole(array_keys(UserGroups::arrayGroup()))) {
            redirect('/admin/main');
        }
    }

    /**
     * Выход
     * @param AdminAuth $authAdmin
     */
    public function logout(AdminAuth $authAdmin): void
    {
        $authAdmin->logOut();
        if (!$authAdmin->check()) {
            redirect('/admin/auth');
        }
    }


    /**
     * Проверка на количество попыток
     * @param $number
     */
    public function numberAttempts($number): void
    {
        if (!empty($number)) {
            $limit_cnt = $this->limit_error - $number;
            if ($number >= 0 && $number <= 8) {
                $this->smarty->assign('error_message', 'auth_wrong');
                $this->smarty->assign('limit_cnt', $limit_cnt);
            } else if ($number === 9) {
//                $this->smarty->assign('message_limit', "Осталась последняя попытка!<br>Советуем не тратить свое время)");
                $this->smarty->assign('message_limit', "There is one last attempt left! We advise you not to waste your time)");
            } else if ($number === 10) {
                Managers::init()->setUserBanned();
                $this->smarty->assign('isBanned', 'banned_message');
            }
        }
    }

    public function fetch()
    {
//        print_r($_SESSION);
        if ($this->authAdmin->isLoggedIn()) {
            $this->redirectRoleAdmin($this->authAdmin);
        }

        try {
            return $this->smarty->fetch('auth/index.tpl');
        } catch (\SmartyException | \Exception $e) {
            print $e->getMessage();
        }
    }

}









