<?php

namespace Mnv\Core;

use JsonException;
use Mnv\Core\Locale\I18N;
use Mnv\Core\Managers\Role;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class Feedback
{
    private $charset;
    private $smtp_server;
    private $smtp_mail;
    private $smtp_user;
    private $smtp_pass;
    private $smtp_secure;
    private $smtp_port;
    private $smtp_debug;

    private $remoteIp;
    private $private_key;

    public $errors;
    public $success = [];

    private $type;
    private $captchaEnabled;
    private $allowCaptcha;

    private ?string $email;
    private ?string $company_name;

    /**
     * Feedback constructor.
     */
    public function __construct($transportMail)
    {
        $this->type = $transportMail;

        $this->charset     = Config::getValue('charset');
        $this->smtp_debug  = Config::getValue('smtp_debug');
        $this->smtp_server = Config::getValue('smtp_server');
        $this->smtp_mail   = Config::getValue('smtp_mail');
        $this->smtp_user   = Config::getValue('smtp_user');
        $this->smtp_pass   = Config::getValue('smtp_pass');
        $this->smtp_secure = Config::getValue('smtp_secure');
        $this->smtp_port   = Config::getValue('smtp_port');

        $this->email            = Config::getValue('email');
        $this->company_name     = Config::getValue('company_name');

        $this->remoteIp    = $_SERVER['REMOTE_ADDR'];

        if (Config::getValue('allow_recaptcha') === 1) {
            $this->private_key = Config::getValue('recaptcha_private_key');
        }

    }

    /**
     * получение параметра, включена капча или нет
     * @param $param
     */
    public function getCaptchaEnabled($param): void
    {
        $this->captchaEnabled = $param;
    }

    /**
     * @return mixed
     */
    public function setCaptchaEnabled()
    {
        return $this->captchaEnabled;
    }

    /**
     * тип капти
     * @param $param
     */
    public function getAllowCaptcha($param): void
    {
        $this->allowCaptcha = $param;
    }

    /**
     * @return mixed
     */
    public function setAllowCaptcha()
    {
        return $this->allowCaptcha;
    }

    /**
     * метод для проверки длины строки
     * @param $string
     * @param $minLength
     * @param $maxLength
     * @return bool
     */
    public function checkStringLength($string, $minLength, $maxLength): bool
    {
        $length = mb_strlen($string, Config::getValue('charset'));
        return !(($length < $minLength) || ($length > $maxLength));
    }

    /**
     * метод проверки имени (name)
     *
     * @param $name
     * @return bool
     */
    public function checkName($name): bool
    {
        if (!empty($name)) {
            if (!$this->checkStringLength($name, 2, 30)) {
                $this->errors = I18N::locale(
                    "* Поля имя содержит недопустимое количество символов!",
                    "* Maydon nomi belgilar nomaqbul sonini o'z ichiga oladi!",
                    "* The name field contains an invalid number of characters!");;
                $this->errors = 'Поля имя содержит недопустимое количество символов!';
                return false;
            }
        } else {
            $this->errors =  I18N::locale(
                "* Не допустимо оставлять поле имя пустым!",
                "* Maydon nomini bo'sh qoldirish joiz emas!",
                "* It is unacceptable to leave the name field empty!"
            );
            return false;
        }
        return true;
    }


    /**
     * проверка капча (reCaptcha:v3)
     *
     * @param $captcha
     * @return bool
     * @throws JsonException
     */
    public function captchaToken($captcha): bool
    {
        if (isset($captcha)) {
            $params = [
                'secret' => $this->private_key,
                'response' => $captcha,
                'remoteip' => $this->remoteIp
            ];
        } else {
            $this->errors = 'Captcha doesn\'t work well. Contact your administrator!';
        }

        $curl = curl_init('https://www.google.com/recaptcha/api/siteverify');
        curl_setopt($curl, CURLOPT_POST, 1);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $params);
        curl_setopt($curl, CURLOPT_FOLLOWLOCATION, 1);
        curl_setopt($curl, CURLOPT_HEADER, 0);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        $response = curl_exec($curl);
        if (!empty($response)) {
            $decoded_response = json_decode($response, false, 512, JSON_THROW_ON_ERROR);
        }
        if ($decoded_response && $decoded_response->success && $decoded_response->score > 0) {
            //$success = $decoded_response->success;
            return true;
        }

        $this->errors = I18N::locale("* Вы не прошли тест", "* Sinovdan o'tolmading.", "* You did not pass the test, I am a robot");
    }

    /**
     * Анти бот
     * @param $bot_one
     * @param $bot_two
     * @param $bot_email
     * @return bool
     */
    public function antiBot($bot_one, $bot_two, $bot_email): bool
    {
        if (trim($bot_one) !== date('Y')){
            if (trim($bot_one) !== trim($bot_two)) {
                $this->errors =  I18N::locale("Я робот! Подобрал капчу", "Men robotman! Oldi captcha", 'I am a robot! Picked up captcha');
            } elseif (empty($bot_two)) {
                $this->errors = I18N::locale("Ошибка: пустой ответ.", "Xato: bo'sh javob.", "Error: empty response.");
            } else {
                $this->errors = I18N::locale("Ошибка: неправильный ответ.", "Xato: noto'g'ri javob.", "Error: wrong answer.");
            }
        } elseif (!empty($bot_email)) {
            $this->errors = 'Bots go no!';
        } else {
            return true;
        }
    }

    /**
     * Отправка письма PHPMailer (smtp)
     *
     * @param $recipient
     * @param $companyName
     * @param string|null $user_email
     * @param string|null $user_name
     * @param $subject
     * @param $htmlBody
     * @param null $attachment
     * @return bool|string
     * @throws Exception
     */
    public function send(?string  $user_email, ?string $user_name, $subject, $htmlBody, $attachment = null)
    {
        $mail = new PHPMailer();
        $mail->CharSet      = $this->charset;
        $mail->Encoding     = "base64";
        $mail->isSMTP();
        $mail->Timeout = 10;
        $mail->SMTPAutoTLS  = false;
        $mail->SMTPDebug    = $this->smtp_debug;
        $mail->Host         = $this->smtp_server;
        $mail->SMTPSecure   = $this->smtp_secure;
        $mail->Port         = $this->smtp_port;

        if ($mail->SMTPSecure === 'ssl') {
            $mail->SMTPOptions = array("ssl" => array("verify_peer" => false, "verify_peer_name" => false));
        }

        if ($mail->SMTPSecure === 'tls') {
            $mail->SMTPOptions = array("tls" => array("verify_peer" => false, "verify_peer_name" => false));
        }

        if ($this->smtp_user) {
            $mail->SMTPAuth = true;
            $mail->Username = $this->smtp_user;
            $mail->Password = $this->smtp_pass;
        }

        if ($this->smtp_mail) {
            $mail->From = $this->smtp_mail;
            $mail->Sender = $this->smtp_mail;
        }

        $mail->setFrom($this->smtp_user, $this->company_name);
        $mail->addAddress($this->email, $this->company_name);

        if (!is_null($user_email)) {
            $mail->addReplyTo($user_email, $user_name);
            $mail->addAddress($user_email, $this->company_name);
        }

        $managers = $this->managersReceiveEmails();
        if (!empty($managers)) {
            foreach ($managers as $manager) {
                $mail->addBCC($manager->email, $this->company_name);
                $mail->addAddress($manager->email, $this->company_name);
            }
        }

        if (!empty($attachment) ) {
            $mail->addAttachment($attachment['tmp_name'], $attachment['name']);
        }

        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body    = $htmlBody;

        if (!$mail->send()) {
            $this->errors = $mail->ErrorInfo;
            return $this->errors;
        }

        return true;
    }

    private function managersReceiveEmails()
    {
        DB::init()->connect()->in('accessLevel', [Role::DEVELOPER, Role::ADMIN]);
        DB::init()->connect()->where('receiveEmails', '=', 0);
        return DB::init()->connect()->table('users')->select('email')->orderBy('userId', 'ASC')->getAll();
    }
    /**
     * Скрыть несколько симоволов в email
     * @param $email
     * @param int $minLength
     * @param int $maxLength
     * @param string $mask
     * @return string
     */
    public function maskEmail($email, $minLength = 3, $maxLength = 10, $mask = "***"): string
    {
        $atPos = strrpos($email, "@");
        $name = substr($email, 0, $atPos);
        $len = strlen($name);
        $domain = substr($email, $atPos);

        if (($len / 2) < $maxLength) $maxLength = ($len / 2);

        $shortenedEmail = (($len > $minLength) ? substr($name, 0, $maxLength) : "");
        return  "{$shortenedEmail}{$mask}{$domain}";
    }

    public function maskPhone($phone)
    {
        return substr_replace($phone,'****',4,13);
    }

    public function insertMail(?string $email, ?string $fullName, $phone, ?string  $subject, $htmlBody): void
    {
        $mails = [
            'fullName'  => $fullName,
            'email'     => $email,
            'phone'     => $phone,
            'subject'   => $subject,
            'message'   => $htmlBody,
            'created'   => gmdate('Y-m-d H:i:s'),

        ];
        DB::init()->connect()->table('feedback')->insert($mails);
    }

}
