<?php

namespace Mnv\Core\Files;

use Mnv\Core\DB;
use Mnv\Core\Mnv;
use Mnv\Core\Files\Image\ImageSizes;
use Mnv\Core\Files\Image\ImageGenerator;

use League\Flysystem\Filesystem;
use League\Flysystem\FileAttributes;
use League\Flysystem\DirectoryAttributes;
use League\Flysystem\FilesystemException;
use League\Flysystem\Local\LocalFilesystemAdapter;
use League\MimeTypeDetection\FinfoMimeTypeDetector;
use League\Flysystem\UnixVisibility\PortableVisibilityConverter;

/**
 * Class FileManager
 * @package Mnv\Models
 */
class FileManager extends Mnv
{

    /** @var string database table */
    private static $_files = 'files';
    /** Images */
    public $ext_img = array( 'jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp' );
    public $ext_copy_img = array( 'svg', 'ico' );
    /** Files */
    public $ext_file = array( 'doc', 'docx', 'rtf', 'pdf', 'xls', 'xlsx', 'txt', 'csv', 'ppt', 'pptx');
    /** Video */
    public $ext_video = array( 'mov', 'mpeg', 'm4v', 'mp4', 'avi', 'mpg', 'wma', "webm" );
    /** Audio */
    public $ext_music = array( 'mp3', 'mpga', 'm4a', 'ac3', 'aiff', 'mid', 'ogg', 'wav' );
    /** Archives */
    public $ext_misc  = array( 'zip', 'rar', 'gz', 'tar', 'dmg' );

    /** @var string[] $_hidden_image_files */
    public static $_hidden_image_files = array('.', '..', '.DS_Store', '.htaccess');

    /** @var string[] $_hidden_image_folders */
    public static $_hidden_image_folders = array('admin', 'large', 'medium', 'small', '.git', '.idea');

    /** @var string[] $_hidden_files */
    public static $_hidden_files = array('.', '..', '.DS_Store', '.htaccess', '.gitattributes', '.gitignore');

    /** @var string[] $_hidden_folders */
    public static $_hidden_folders = array('vendor', 'large', 'medium', 'small', '.git', '.idea');

    /** @var string[] $sizes */
    private $sizes = array('large', 'medium', 'small');

    /** @var Filesystem */
    public $filesystem;

    /** @var FinfoMimeTypeDetector  */
    public $detector;

    public $maxSize;

    /**
     * FileManager constructor.
     * @param string|null $path
     */
    public function __construct(?string $path)
    {
        parent::__construct();

        $adapter = new LocalFilesystemAdapter($path,
            PortableVisibilityConverter::fromArray([
                'file' => [
                    'public' => 0666,
                    'private' => 0644
                ],
                'dir' => [
                    'public' => 0777,
                    'private' => 0755
                ]
            ]),
            LOCK_EX,
            LocalFilesystemAdapter::DISALLOW_LINKS
        );

        $this->filesystem = new Filesystem($adapter, ['directory_visibility' => 'public', 'visibility' => 'public']);
        $this->detector = new FinfoMimeTypeDetector();

    }

    /**
     * Загрузка файла
     *
     * @param $path
     * @param array|null $file
     * @param int $managerId
     * @return bool
     * @throws FilesystemException
     */
    public function fileUpload($path, ?array $file, int $managerId): bool
    {
        $post_max_size          = $this->getSizeLimit(\ini_get('post_max_size'));
        $upload_max_filesize    = $this->getSizeLimit(\ini_get('upload_max_filesize'));
//        print_r($post_max_size);	//8388608 - 16 777 216
//        print_r(PHP_EOL);
//        print_r($upload_max_filesize); //67108864 - 67 108 864
//        print_r(PHP_EOL);
//        print_r($_SERVER["CONTENT_LENGTH"]); //15278259 - 15 278 259
        if (($_SERVER["CONTENT_LENGTH"] > $post_max_size) || ($_SERVER["CONTENT_LENGTH"] > $upload_max_filesize)) {
            $this->response = array('status' => 500, 'message' => 'Загруженный файл превышает параметр post_max_size или upload_max_filesize в php.ini', 'error' => 'Maximum allowed size of data sent by the POST method');
            return false;
        }

        if (!empty($file)) {
//            print_r($file);
////           $file = $file['file'];

            $generator = new ImageGenerator($this->filesystem, $managerId);

            if ($path == '') {
                $this->response = array('status' => 500, 'message' => lang('fileManager:errors:18'), 'error' => lang('fileManager:errors:18'));
                return false;
            }

                if ($this->filesystem->fileExists($file['file']['name'])) {
                    $this->response = array('status' => 500, 'message' => lang('fileManager:errors:16'), 'error' => lang('fileManager:errors:16'));
                    return false;
                }

                $stream = fopen($file['file']['tmp_name'], 'r+');
                try {
                    $this->filesystem->writeStream($file['file']['name'], $stream);
                } catch (FilesystemException $e) {
                    $this->response = array('status' => 500, 'message' => $e->getMessage(), 'error' => $e->getMessage());
                    return false;
                }
                if (is_resource($stream)) {
                    fclose($stream);
                }

                $mimeType = $this->detector->detectMimeType($file['file']['name'], '');
                $fileSize = $this->filesystem->fileSize($file['file']['name']);

                if (!$fileSize) {
                    $this->response = array('status' => 500, 'message' => lang('fileManager:errors:13'), 'error' => lang('fileManager:errors:13') . ' (FILE SIZE EMPTY)');
                    if ($this->filesystem->fileExists($file['file']['name'])) $this->filesystem->delete($file['file']['name']);
                    return false;
                }

                $mimeTypeArr = explode('/', $mimeType);
                if ($mimeType) {
                    $file['file']['extension'] = $mimeTypeArr[1];
                    $file['file']['directory'] = $path;

                    if ((int)$this->config['max_file_size'] && $fileSize > ($this->config['max_file_size'] * 1024)) {
                        $this->response = array('status' => 500, 'message' => lang('fileManager:errors:14'), 'error' => lang('fileManager:errors:14') . ' (MAX FILE SIZE)');
                        if ($this->filesystem->fileExists($file['file']['name'])) {
                            $this->filesystem->delete($file['file']['name']);
                        }
                        return false;
                    }

                    if ($mimeTypeArr[0] === 'image' && $mimeTypeArr[1] != 'svg') {

                        if ((int)$this->config['max_up_size'] && $fileSize > ($this->config['max_up_size'] * 1024)) {
                            $this->response = array('status' => 500, 'message' => lang('fileManager:errors:15'), 'error' => lang('fileManager:errors:15') . ' (MAX UP SIZE)');
                            if ($this->filesystem->fileExists($file['file']['name'])) {
                                $this->filesystem->delete($file['file']['name']);
                            }
                            return false;
                        }

                        $uploaded = $generator->init($file['file'], $fileSize, $mimeType)->apply();

                    } else {
                        $uploaded = $generator->safeFile($file['file'], str_replace("/", "", $path), '/uploads/' . $path);
                    }

                } else {
                    $uploaded = $generator->safeFile($file['file'], str_replace("/", "", $path), '/uploads/' . $path);
                }

                if (!$uploaded) {
                    if ($this->filesystem->fileExists($file['file']['name'])) {
                        $this->filesystem->delete($file['file']['name']);
                    }
                    $this->response = array('status' => 500, 'message' => lang('fileManager:errors:17'), 'error' => lang('fileManager:errors:17'));
                    return false;
                }

                $this->response = array('status' => 200, 'message' => lang('fileManager:messages:1'), 'error' => '', 'fileId' => $uploaded);
                return true;
            }

        $this->response = array('status' => 500, 'message' => 'Загруженный файл превышает параметр post_max_size в php.ini', 'error' => 'Maximum allowed size of data sent by the POST method');
        return false;


    }

    /**
     * Список файлов
     *
     * @param array|null $where
     * @param string|null $query
     * @param string|null $filters
     * @param string|null $type
     * @param int $managerId
     * @return array|mixed
     */
    public function listContents(?array $where, ?string $query, ?string $filters, ?string $type, int $managerId)
    {
        $list = array();
        if (!empty($type)) {
            if ($type == 'mine') {
                $where['addedBy'] = $managerId;
            } else {
                DB::init()->connect()->where('mime_type', $type);
            }
        }
        if (!empty($query))
            DB::init()->connect()->like('fileName', "%$query%");

        if (isset($filters) && !empty($filters)) {
            if ($filters == 'date') DB::init()->connect()->orderBy('addedOn DESC');
            if ($filters == 'size') DB::init()->connect()->orderBy('size ASC');
            if ($filters == 'name') DB::init()->connect()->orderBy('fileName ASC');
        } else {
            DB::init()->connect()->orderBy('fileId DESC');
        }

        $lists = DB::init()->connect()->table(static::$_files)->where($where)->pagination($this->limit, $this->page)->getAll();
//        print_r(DB::init()->connect()->getQuery());
        if ($lists) {
            foreach ($lists as $k => $file) {
                $extension = $this->fileExtension($file->fileName);
                $list['files'][$k] = array(
                    'fileId'    => $file->fileId,
                    'fileName'  => $file->fileName,
                    'size'      => $this->formatFileSize($file->size),
                    'icon'      => GLOBAL_URL . '/admin/assets/ico/' . strtolower($extension) . '.svg',
                    'original'  => GLOBAL_URL . $file->path . $file->fileName
                );

                /** Video */
                if (in_array(strtolower($extension), $this->ext_video)) {
                    $list['files'][$k]['class_ext'] = 4;
                }
                /** Images */
                elseif (in_array(strtolower($extension), $this->ext_img)) {
                    $list['files'][$k]['class_ext'] = 2;
                }
                /** Images svg */
                elseif (in_array(strtolower($extension), $this->ext_copy_img)) {
                    $list['files'][$k]['class_ext'] = 2;
                }
                /** Audio */
                elseif (in_array(strtolower($extension), $this->ext_music)) {
                    $list['files'][$k]['class_ext'] = 5;
                }
                /** Archives */
                elseif (in_array(strtolower($extension), $this->ext_misc)) {
                    $list['files'][$k]['class_ext'] = 3;
                } else {
                    $list['files'][$k]['class_ext'] = 1;
                }
            }

            return $list['files'];
        }

        return $list;
    }

    /**
     * Получение файла
     *
     * @param $realPath
     * @param $fileId
     * @return array
     */
    public function getFile($realPath, $fileId): array
    {
        $list = array();
        if ($file = DB::init()->connect()->table(static::$_files)->where('fileId', $fileId)->get()) {
            $extension = strtolower($this->fileExtension($file->fileName));
            $list['fileId']     = $file->fileId;
            $list['extension']  = $extension; // нужно для видео файлов
            $list['fileName']   = $file->fileName;
            $list['size']       = $this->formatFileSize($file->size);
            $list['url']        = $file->path . $file->fileName;
            $list['icon']       = GLOBAL_URL.'/admin/assets/ico/'.$extension.'.svg';

            if (in_array($extension, $this->ext_video)) { /** Video */
                $list['class_ext'] = 4;
            } elseif (in_array($extension, $this->ext_img)) { /** Image */
                $list['class_ext'] = 2;
            } elseif (in_array($extension, $this->ext_copy_img)) { /** Images svg */
                $list['class_ext'] = 2;
            } elseif (in_array($extension, $this->ext_music)) { /** Audio */
                $list['class_ext'] = 5;
            } elseif (in_array($extension,  $this->ext_misc)) { /** Archive zip, rar */
                $list['class_ext']  = 3;
            } else {
                $list['class_ext']  = 1;
                $list['original']   = $realPath . $file->fileName;
            }
        }
        $this->response = array('status' => 200, 'message' => '', 'error' => '', 'list' => $list);
        return $list;
    }

    /**
     * Получение загруженных директорий по указанному пути
     * используется в `UploadsAdmin`
     *
     * @param $path
     * @param false $recursive
     * @return array|array[]
     * @throws FilesystemException
     */
    public function ListDirectories($path, bool $recursive = false): array
    {
        $listing = array();
        $array = array('dirs' => array());

        try {
            $listing = $this->filesystem->listContents($path, $recursive)->sortByPath();
        } catch (FilesystemException $e) {
            print_r($e->getMessage());
        }

        foreach ($listing as $item) {
            if ($path == $item->path()) continue;
            $path_info = $item->path();

            $finfo = pathinfo($path_info);
            $name = $finfo['basename'];

            if ($item instanceof DirectoryAttributes) {
                if (!in_array($name, self::$_hidden_image_folders)) {
                    list($size, $files_count, $folders_count) = $this->determineDirectorySize($name);
                    $array['dirs'][] = [
                        'path' => $path_info,
                        'name' => $name,
                        'mtime' => $item->lastModified(),
                        'size'          => $this->formatFileSize($size),
                        'file_count'    => $files_count,
                        'folders_count' => $folders_count,
                    ];
                }
            }
        }

        return $array;
    }

    /**
     * Получение всех директорий и файлов
     * используется в `FileManagerAdmin`
     *
     * @param $path
     * @param false $recursive
     * @return array|array[]
     */
    public function ListDirectory($path, bool $recursive = false): array
    {
        $listing = array();
        $array = array('dirs' => array(), 'files' => array());

        try {
            $listing = $this->filesystem->listContents($path, $recursive)->sortByPath();
        } catch (FilesystemException $e) {
            print_r($e->getMessage());
        }

        foreach ($listing as $item) {
            if ($path == $item->path()) continue;
            $path_info = $item->path();

            $finfo = pathinfo($path_info);
            $name = $finfo['basename'];
            /** файлы */
            if ($item instanceof FileAttributes) {
                if (!in_array($name, self::$_hidden_files)) {
                    $array['files'][] = [
                        'path' => $path_info,
                        'name' => $name,
                        'size' => $item->fileSize(),
                        'mtime' => $item->lastModified()
                    ];
                }

            }
            /** директории */
            elseif ($item instanceof DirectoryAttributes) {
                if (!in_array($name, self::$_hidden_folders)) {
                    $array['dirs'][] = [
                        'path' => $path_info,
                        'name' => $name,
                        'mtime' => $item->lastModified()
                    ];
                }
            }
        }

        return $array;
    }

    /**
     * Определение размера каталога / Determine directory size
     * determineDirectorySize
     *
     * @param string $path
     * @param bool $recursive
     * @return array
     * @throws FilesystemException
     */
    private function determineDirectorySize(string $path, bool $recursive = false): array
    {
        $total_size = 0;
        $files_count = 0;
        $folders_count = 0;
        $listing = array();

        try {
            $listing = $this->filesystem->listContents($path, $recursive)->sortByPath();
        } catch (FilesystemException $e) {
            print_r($e->getMessage());
        }

        foreach ($listing as $item) {
            if ($path == $item->path()) continue;
            $path_info = $item->path();

            $finfo = pathinfo($path_info);
            $name = $finfo['basename'];
            if ($item instanceof FileAttributes) {
                if (!in_array($name, self::$_hidden_image_files)) {
                    $size = $this->filesystem->fileSize($item->path());
                    $total_size += $size;
                    $files_count++;
                }

            } elseif ($item instanceof DirectoryAttributes) {
                if (!in_array($name, self::$_hidden_image_folders)) {
                    list($size) = $this->determineDirectorySize($name);
                    $total_size += $size;
                    $folders_count ++;
                }
            }
        }

        return array($total_size, $files_count, $folders_count);
    }


    /**
     * Сортировать по алфавиту
     *
     * @param $a
     * @param $b
     * @return int
     */
    public function sortListByName(&$a, &$b): int
    {
        return strcasecmp($a['name'], $b['name']);
    }

//    /**
//     * Удаление файла
//     *
//     * @param $fileId
//     * @return bool
//     * @throws FilesystemException
//     */
//    public function deleteFile($fileId): bool
//    {
//        if ($file = DB::init()->connect()->table(self::$_files)->where('fileId', $fileId)->get()) {
//            $this->filesystem->delete($file->fileName);
//            foreach ($this->sizes as $size) {
//               $this->filesystem->delete( $size . '/' .$file->fileName);
//            }
//            DB::init()->connect()->table(static::$_files)->where('fileId', $file->fileId)->delete();
//
//            return true;
//        }
//
//        return false;
//    }

    /**
     * Удаление файла / файлов
     *
     * @param $fileIds
     * @return bool
     * @throws FilesystemException
     */
    public function deleteFiles($fileIds): bool
    {
        if ($files = DB::init()->connect()->table(static::$_files)->in('fileId', $fileIds)->getAll()) {
            foreach ($files as $file) {
                $this->filesystem->delete($file->fileName);
                foreach ($this->sizes as $size) {
                    $this->filesystem->delete($size . '/' . $file->fileName);
                }

                DB::init()->connect()->table(static::$_files)->where('fileId', $file->fileId)->delete();
            }
            return true;
        }

        return false;
    }

    /**
     * Удаление всех файлов с указаной директорией
     *
     * @param string $path
     * @return bool
     */
    public function deleteAllFilesThisDirectory(string $path): bool
    {
        if ($files = DB::init()->connect()->table(static::$_files)->where('folder', $path)->getAll()) {
            foreach ($files as $file) {
                DB::init()->connect()->table(static::$_files)->where('fileId', $file->fileId)->delete();
            }

            return true;
        }

        return false;
    }

    /**
     * получение название последнего елемента и строки (будет название папки)  по данному пути `$path`
     *
     * @param string $path
     * @return false|mixed|string
     */
    public function getFolderName(string $path)
    {
        $folder = explode('/', $path);
        $folder = array_diff($folder, array(''));
        return end($folder);
    }

    /**
     * Получить расширение файла
     *
     * @param $name
     * @return false|string
     */
    public function fileExtension($name)
    {
        $n = strrpos($name, '.');
        return ($n === false) ? '' : substr($name, $n+1);
    }

    /**
     *
     * @param $file_size
     * @return string
     */
    private function formatFileSize($file_size): string
    {
        if (!$file_size OR $file_size < 1) return '0 b';

        $prefix = array("b", "Kb", "Mb", "Gb", "Tb");
        $exp = floor(log($file_size, 1024)) | 0;

        return round($file_size / (pow(1024, $exp)), 2).' '.$prefix[$exp];
    }

    /**
     * Метод для получения картинки при загрузки в контент (tinymce)
     *
     * @param $fileId
     * @return string|null
     */
    public function getFileUrl($fileId): ?string
    {
        if ($file = DB::init()->connect()->table(static::$_files)->where('fileId', $fileId)->get()) {
            $image  = ImageSizes::init()->get(null, $file);
            return $image->original ?? '';
        }

        return null;
    }

    protected function getSizeLimit($limit)
    {
        $limit = $this->sizeToBytes($limit);
        if ($this->maxSize !== null && $limit > 0 && $this->maxSize < $limit) {
            $limit = $this->maxSize;
        }
        if (isset($_POST['MAX_FILE_SIZE']) && $_POST['MAX_FILE_SIZE'] > 0 && $_POST['MAX_FILE_SIZE'] < $limit) {
            $limit = $_POST['MAX_FILE_SIZE'];
        }
        return $limit;
    }

    public function sizeToBytes($sizeStr)
    {
        switch (strtolower(substr($sizeStr, -1)))
        {
            case 'm': return (int)$sizeStr * 1048576; // 1024 * 1024
            case 'k': return (int)$sizeStr * 1024; // 1024
            case 'g': return (int)$sizeStr * 1073741824; // 1024 * 1024 * 1024
            default: return (int)$sizeStr; // do nothing
        }
    }



}