<?php

namespace Mnv\Models\Users;

use Mnv\Core\DB;
use Mnv\Core\Managers\Role;
use Mnv\Models\Users\Exceptions\NoUserException;
use Mnv\Models\Users\Exceptions\EmptyUserIdException;
use Mnv\Models\Users\Exceptions\NotUpdateUserException;
use Mnv\Models\Users\Exceptions\UserNotDeletedException;
use Mnv\Models\Users\Exceptions\NotUserApproveException;
use Mnv\Models\Users\Exceptions\CannotBeRemovedDeveloperException;
use Mnv\Models\Users\Exceptions\CannotBeRemovedAdministratorException;

/**
 * Class Managers
 */
class AdminUser extends AbstractUser implements UserInterface
{

    /** @var array|string права доступа */
    public $accessLevels;
    /** @var int $totalArticle кол-во контента сланного менеджером  */
    public $totalArticle;

    /** ManagersAdmin constructor
     * @throws \SmartyException
     */
    public function __construct()
    {

        parent::__construct();
    }



    /**
     * Получение всех менеджеров
     *
     * @param string|null $query (optional)
     * @param string|null $status (optional)
     * @param string|null $orderBy (optional)
     * @param $userTypes
     * @param $manager
     */
    public function getAll(?string $query, ?string $status, ?string $orderBy, $userTypes, $manager = null) : void
    {

        $this->getUsers($query, $status,'adminUser', $orderBy, $userTypes, $manager);

    }

    /**
     * получение кол-во пользователей
     *
     * @param string|null $query
     * @param string|null $status
     * @param $userTypes
     * @param $manager
     */
    public function total(?string $query, ?string $status, $userTypes, $manager = null) : void
    {
        $this->getCountUsers($query, $status, 'adminUser', $userTypes, $manager);
    }


    /**
     * Редактирование
     * @throws NoUserException
     */
    public function edit(?string $userType)
    {
        if (!empty($this->userId)) {
            $this->getUser($userType);
            if (!empty($this->user)) {
                $this->user['registered'] = gmdate('Y-m-d H:i', $this->user['registered']);
                $this->user['last_login'] = gmdate('Y-m-d H:i', $this->user['last_login']);

                $this->user['image'] = $this->getFile($this->user['fileId']);
                $this->user['accessLevelName'] = DB::init()->connect()->table('user_group')->select('groupName')->where('usergroup_id', $this->user['accessLevel'])->getValue();

                $this->totalArticle = DB::init()->connect()->table('articles')->count('*', 'count')->where('addedBy', $this->user['userId'])->getValue();

                $this->user['notice'] = $this->notice();
            } else {
                throw new NoUserException();
            }
        }
    }

    private function notice(): string
    {
        if ($res =  DB::init()->connect()->table('notice')->where('userId', $this->user['userId'])->get()) {
            return htmlspecialchars( $res->notice, ENT_QUOTES, $this->config['charset'] );
        }

        return '';
    }

    /**
     * Создание нового менеджера
     *
     * @param $user
     * @param int $managerId
     * @return bool
     */
    public function add($user, int $managerId): bool
    {
        $userInsert = array_filter([
            'userType'      => $user['userType'],
            'fullName'      => $user['fullName'],
            'loginName'     => $user['loginName'],
            'email'         => $user['email'],
            'password'      => \password_hash($user['password'], \PASSWORD_DEFAULT),
            'phone'         => $user['phone'],
            'address'       => $user['address'],
            'accessLevel'   => $user['accessLevel'],
            'status'        => $user['status'],
            'receiveEmails' => $user['receiveEmails'],
            'fileId'        => $user['fileId'],
            'addedBy'       => $managerId,
            'addedOn'       => gmdate('Y-m-d H:i:s'),
            'modifiedBy'    => $managerId,
            'modifiedOn'    => gmdate('Y-m-d H:i:s'),
            'registered'    => \time(),
            'last_login'    => \time(),
            'verified'      => 1,
        ]);

        if ($this->insertUser((array)$userInsert)) {
            return true;
        }

        return false;
    }

    /**
     * Обновление менеджера
     *
     * @param $user
     * @param int $managerId
     * @return bool
     * @throws NotUpdateUserException
     */
    public function update($user, int $managerId): bool
    {
        if (!empty($this->userId)) {
            $userUpdate = array_filter([
                'userType'      => $user['userType'],
                'fullName'      => $user['fullName'],
                'loginName'     => $user['loginName'],
                'email'         => $user['email'],
                'phone'         => $user['phone'],
                'accessLevel'   => $user['accessLevel'],
                'status'        => $user['status'],
                'fileId'        => $user['fileId'],
                'modifiedBy'    => $managerId,
                'modifiedOn'    => gmdate('Y-m-d H:i:s'),
            ]);

            if (!empty($user['newPassword'])) {
                $userUpdate['password'] = \password_hash($user['newPassword'], \PASSWORD_DEFAULT);
            }
            $userUpdate['receiveEmails'] = !empty($user['receiveEmails']) ? 1 : 0;


            if ($this->updateUser($userUpdate)) {

                $this->getUser($user['userType']);
                $this->user['image'] = $this->getFile($this->user['fileId']);
                $this->user['accessLevelName'] = lang('managers:accessRole:' . $this->user['accessLevel']);

                return true;
            }

            throw new NotUpdateUserException();
        }
        return false;
    }

    /**
     * Права доступа / Access rights
     *
     * @param $accessLevelId
     * @return array
     */
    public function accessRights($accessLevelId): array
    {
        $permission = json_decode(file_read(GLOBAL_ROOT."/admin/menu/menu.json"), true);
        $result = array();

        foreach ($permission as $key => $item) {
            if (!in_array($accessLevelId, $item['accessLevel']))  continue;
            $result[$key] = $item;
            $child = array();
            foreach ($result[$key]['children'] as $childrenKey => $children) {
                if (!in_array($accessLevelId, $children['accessLevel'])) continue;
                $child[$childrenKey] = $children;
            }
            unset( $result[$key]['children']);

            $result[$key]['children'] = $child;
        }

        return $result;
    }


    /**
     * Удаление пользователя
     *
     * @throws NoUserException
     * @throws EmptyUserIdException
     * @throws UserNotDeletedException
     * @throws CannotBeRemovedDeveloperException
     * @throws CannotBeRemovedAdministratorException
     */
    public function remove(?string $userType)
    {
        if ( $this->userId !== null ) {
            $this->getUser($userType);
            if ($this->user !== null) {
                if ( $this->user['userType'] == UserTypes::DEVELOPER ) {
                    if ( !$this->getUserByRole(UserTypes::DEVELOPER) ) {
                        throw new CannotBeRemovedDeveloperException();
                    }
                } else if ( $this->user['userType'] == UserTypes::ADMIN ) {
                    if ( !$this->getUserByRole(UserTypes::ADMIN) ) {
                        throw new CannotBeRemovedAdministratorException();
                    }
                }
                if ( !$this->removeUser() ) {
                    throw new UserNotDeletedException();
                }
            } else {
                throw new NoUserException();
            }
        } else {
            throw new EmptyUserIdException();
        }
    }

    /**
     * Обновление статуса
     *
     * @throws NoUserException
     * @throws EmptyUserIdException
     * @throws NotUserApproveException
     */
    public function approve()
    {
        if ( $this->userId !== null ) {
            $this->getUser();
            if ($this->user !== null) {
                if (!$this->approveUser()) {
                    throw new NotUserApproveException();
                }
            }else {
                throw new NoUserException();
            }
        } else {
            throw new EmptyUserIdException();
        }
    }



}