<?php

namespace Mnv\Models\Contents;

use Mnv\Core\DB;
use Mnv\Core\Managers;
use Mnv\Models\Contents\Exceptions\NoContentException;
use Mnv\Models\Contents\Exceptions\EmptyArticleIdException;
use Mnv\Models\Contents\Exceptions\NotInsertContentException;
use Mnv\Models\Contents\Exceptions\NotUpdateContentException;
use Mnv\Models\Contents\Exceptions\NotContentApproveException;
use Mnv\Models\Contents\Exceptions\ContentNotDeletedException;


/**
 * Class Content
 * @package Mnv\Models\Contents
 */
class Content extends AbstractContent
{

    private static $_table            = 'articles';
    private static $_table_images     = 'article_images';
    private static $_table_comments   = 'comments';
    private $_fields = array(
        'articleId',
        'sectionId',
        'title',
        'alias',
        'url',
        'publishedOn',
        'orderBy',
        'isFeatured',
        'status',
        'typeContent',
        'typeNews',
        'enableGallery',
        'blocks'
    );

    /**
     * Получение всех статей
     *
     * @param string|null $orderBy
     */
    protected function listContent(?string $orderBy = 'publishedOn DESC'): void
    {
        /** фильтрация / сортировка */
        $this->sorting();

        $this->contents = DB::init()->connect()->table(self::$_table)->select($this->_fields)->orderBy($orderBy)->indexKey('articleId')->pagination($this->limit, $this->page)->getAllIndexes('array');

        $this->total();
    }

    /**
     * Получение кол-во контента
     */
    protected function total(): void
    {
        /** фильтрация / сортировка */
        $this->sorting();

        $this->total = (int)DB::init()->connect()->table(self::$_table)->count('*', 'count')->getValue();
    }

    /**
     * Получить контент
     *
     */
    protected function getContent() : void
    {
        $this->content = DB::init()->connect()->table(self::$_table)->where('articleId',  $this->articleId)->get('array');
    }

    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $fileName
     * @return int|mixed|string|null
     */
    protected function checkFileName(?string $fileName)
    {
        if (empty($fileName)) {
            $maxId = DB::init()->connect()->table(self::$_table)->max('articleId')->getValue();
            return $maxId ? $maxId + 1 : 1;
        }

        if (empty($this->articleId)) {
            if ($fileName = DB::init()->connect()->table(self::$_table)->select('fileName')->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
                return $fileName;
            }
        }
        else if ($fileName = DB::init()->connect()->table(self::$_table)->select('fileName')->where('articleId','<>', $this->articleId)->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
            return $fileName;
        }

        return null;
    }

    /**
     * Редактирование и добавление контента
     * @throws NoContentException
     */
    protected function edit(): void
    {
        if (!empty($this->articleId)) {
            $this->getContent();
            if ($this->content !== null) {
                Managers::gettingInfoAboutUsers($this->content);
                /** получение картинок контента */
                $this->getFiles();
            } else {
                throw new NoContentException();
            }
        }

    }

    /**
     * Добавление нового контента
     *
     * @param array|null $content
     * @param array|null $images
     * @param int $managerId
     * @return mixed|void
     * @throws NotInsertContentException
     */
    protected function add(?array $content, ?array $images, int $managerId)
    {
//        $content['votes'] = 0;

        $content['addedBy']     = $managerId;
        $content['addedOn']     = gmdate('Y-m-d H:i:s');
        $content['modifiedBy']  = $managerId;
        $content['modifiedOn']  = gmdate('Y-m-d H:i:s');
        $content['publishedOn'] = adjustTime(date('Y-m-d H:i:s', strtotime($content['publishedOn'])),  true);
        $content['orderBy']     = DB::init()->connect()->table(self::$_table)->max('orderBy')->where('typeContent', $content['typeContent'])->getValue() + 1;

        /* insert новую запись и возвращаем ID */
        if ($this->articleId = (int)DB::init()->connect()->table(self::$_table)->insert((array)$content)) {
            if (!empty($images['fileId'])) {
                $this->general($this->articleId, $images);
            }
            if (!empty($images['gallery'])) {
                $this->picture($this->articleId, array_keys($images['gallery']));
            }
            if (!empty($images['docs'])) {
                $this->docs($this->articleId, $images);
            }
        } else {
            throw new NotInsertContentException();
        }

    }


    /**
     * Обновление контента
     *
     * @param array|null $content
     * @param array|null $images
     * @param int $managerId
     * @return mixed|void
     * @throws NotUpdateContentException
     */
    protected function update(?array $content, ?array $images, int $managerId)
    {
        $content['modifiedBy']   = $managerId;
        $content['modifiedOn']   = gmdate('Y-m-d H:i:s');
        $content['publishedOn']  = adjustTime(date('Y-m-d H:i:s', strtotime($content['publishedOn'])),  true);

        if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->update($content)) {

            if (!empty($images['fileId'])) {
                $this->general($this->articleId, $images);
            }

            if (!empty($images['gallery'])) {
                $this->picture($this->articleId, array_keys($images['gallery']));
            }

            if (!empty($images['docs'])) {
                $this->docs($this->articleId, $images);
            }

            $this->getContent();
            /* получать информацию о пользователе, который создал или редактировал раздела */
            Managers::gettingInfoAboutUsers($this->content);
            /* получение картинок */
            $this->getFiles();
        } else {

            throw new NotUpdateContentException();
        }
    }

    /**
     * Удаление контента
     *
     * @throws EmptyArticleIdException
     * @throws ContentNotDeletedException
     * @throws NoContentException
     */
    protected function remove(): bool
    {
        if ($this->articleId !== null) {
            $this->getContent();
            if ($this->content !== null) {
                if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->delete() ) {
                    DB::init()->connect()->table(self::$_table_comments)->where('articleId', $this->articleId)->delete();
                    DB::init()->connect()->table(self::$_table_images)->where('articleId', $this->articleId)->delete();

                    return true;
                }
                throw new ContentNotDeletedException();
            }

            throw new NoContentException();
        }

        throw new EmptyArticleIdException();
    }

    /**
     * Обновление статуса
     *
     * @throws EmptyArticleIdException
     * @throws NoContentException
     * @throws NotContentApproveException
     */
    protected function approve(): bool
    {
        if ( $this->articleId !== null ) {
            $this->getContent();
            if ($this->content !== null) {
                if (DB::init()->connect()->table(self::$_table)->where('articleId',$this->articleId)->update(['status' => 'visible'])) {
                    return true;
                }
                throw new NotContentApproveException();
            }

            throw new NoContentException();
        }

        throw new EmptyArticleIdException();
    }

    /**
     * Изменение статуса
     *
     * @throws NoContentException
     * @throws EmptyArticleIdException
     * @throws NotContentApproveException
     */
    protected function status(): void
    {
        if ( $this->articleId !== null ) {
            $this->getContent();
            if ($this->content !== null) {
                $contentUpdate['status'] = ($this->content['status'] === 'visible') ? 'hidden' : 'visible';
                if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->update($contentUpdate)) {
                    $this->result = array('data' => true, 'status' => $contentUpdate['status']);
                    return;
                }

                throw new NotContentApproveException();
            }

            throw new NoContentException();
        }

        throw new EmptyArticleIdException();
    }

    /**
     * Сортировка контента
     *
     * @param $ids
     * @param $articleIds
     */
    protected function reorder($ids, $articleIds): void
    {
        foreach ($articleIds as $i => $articleId) {
            DB::init()->connect()->table(self::$_table)->where('articleId','=',$ids[$i])->update(['orderBy' => $articleId]);
        }
    }

    /**
     * Групповые действия
     * @param $group_action
     */
    protected function group($group_action): void
    {
        if ($group_action === 'status') {
            $this->getContent();
            if ($this->content !== null) {
                $contentUpdate['status'] = ($this->content->status === 'visible') ? 'hidden' : 'visible';
                $this->statusContent($contentUpdate);
            }

        } else if ($group_action === 'remove') {
            $this->getContent();
            if ($this->content !== null) {
                $this->removeContent();
            }
        }
    }

    /**
     * Обновление статуса контента
     *
     * @param $contentUpdate
     * @return bool
     */
    protected function statusContent($contentUpdate): bool
    {
        if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->update($contentUpdate)) {
            return true;
        }

        return false;
    }

    /**
     * Удаление контента
     */
    protected function removeContent(): bool
    {
        /** удаление контента и удаление записей из базы данных прикрепленных картинок и комментариев к этому контенту */
        if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->delete()) {
            DB::init()->connect()->table(self::$_table_comments)->where('articleId', $this->articleId)->delete();
            DB::init()->connect()->table(self::$_table_images)->where('articleId', $this->articleId)->delete();

            return true;
        }

        return false;
    }


    /**
     * Удаление основной картинки
     * @param $imageId
     * @return bool
     */
    protected function deleteImage($imageId): bool
    {
        if (!empty($imageId)) {
            DB::init()->connect()->table(self::$_table_images)->where('imageId',  $imageId)->delete();
            return true;
        }

        return false;
    }

    /**
     * Добавление или замена информации у картинки в галереи
     *
     * @param $imageId
     * @param $picture
     * @return array|false|int
     */
    protected function editPictureInfo($imageId, $picture)
    {
        $updateParams = [
            'title'         => $picture['title'],
            'alias'         => $picture['alias'],
            'description'   => $picture['description'],
            'link'          => $picture['link'],
            'position'      => $picture['position']
        ];

        if (DB::init()->connect()->table(self::$_table_images)->where('imageId', $imageId)->update($updateParams)) {
            return DB::init()->connect()->table(self::$_table_images)->where('imageId', $imageId)->get();
        }
        return false;
    }

    /**
     * Сортировка картинок в галереи
     * @param $imageId
     * @param $orderBy
     * @return bool
     */
    protected function sortPictures($imageId, $orderBy): bool
    {
        if (DB::init()->connect()->table(self::$_table_images)->where('imageId', $imageId)->update(['orderBy' => $orderBy])) {
            return true;
        }


        return false;
    }

    /**
     * Удаление картинки из галереи
     * @param $imageId
     * @return bool
     */
    protected function deletePicture($imageId): bool
    {
        if (DB::init()->connect()->table(self::$_table_images)->where('imageId', $imageId)->delete()) {
            return true;
        }

        return false;
    }

    /**
     * Удаление картинки из галереи
     * @param $imageId
     * @return bool
     */
    public function deleteDocs($imageId): bool
    {
        if (DB::init()->connect()->table(self::$_table_images)->where('imageId',  $imageId)->delete()) {
            return true;
        }

        return false;
    }

    /**
     * Добавление основной картинки к контенту
     *
     * @param $articleId
     * @param $images
     * @return bool
     */
    protected function general($articleId, $images): bool
    {
        $imageUpload = array_filter([
            'articleId' => $articleId,
            'fileId'    => $images['fileId'],
            'type'      => 'general',
            'orderBy'   => 0
        ]);

        if ($imageId = DB::init()->connect()->table(self::$_table_images)->select('imageId')->where('type','general')->where('articleId', $articleId)->getValue()) {
            $imageUpload['imageId'] = $imageId;
        }

        if (DB::init()->connect()->table(self::$_table_images)->replace($imageUpload)) {
            return true;
        }

        return false;
    }

    /**
     * Загрузка картинок в галерею
     *
     * @param $articleId
     * @param $images
     */
    protected function picture($articleId, $images): void
    {
        $idx = 0;
        if ($order = DB::init()->connect()->table(self::$_table_images)->max('orderBy')->where('articleId', $articleId)->getValue()) {
            $idx = $order;
        }

        foreach ($images as $fileId) {
            $idx++;
            $imageUpload = array_filter(['articleId' => $articleId, 'fileId' => $fileId, 'type' => 'gallery', 'orderBy' => $idx]);
            DB::init()->connect()->table(self::$_table_images)->replace($imageUpload);
        }
    }

    /**
     * Добавление файлов
     *
     * @param $sectionId
     * @param $images
     * @return bool
     */
    protected function docs($articleId, $images): bool
    {
        if (!empty($images['docs'])) {
            if ($order = DB::init()->connect()->table(self::$_table_images)->where('articleId', $articleId)->max('orderBy')->getValue()) {
                $idx = $order;
            } else {
                $idx = 0;
            }
            foreach ($images['docs'] as $gallery) {
                $idx++;
                $file = DB::init()->connect()->table('files')->where('fileId', $gallery['fileId'])->get();

                $imageUpload['articleId'] = $articleId;
                $imageUpload['fileId']    = $file->fileId;
                $imageUpload['type']      = 'docs';
                $imageUpload['orderBy']   = $idx;

                DB::init()->connect()->table(self::$_table_images)->replace($imageUpload);
            }
            return true;
        }

        return false;
    }



}