<?php

namespace Mnv\Models\Contents;

use Mnv\Core\DB;
use Mnv\Core\Managers;
use Mnv\Models\Contents\Exceptions\NoContentException;
use Mnv\Models\Contents\Exceptions\EmptyArticleIdException;
use Mnv\Models\Contents\Exceptions\NotInsertContentException;
use Mnv\Models\Contents\Exceptions\NotUpdateContentException;
use Mnv\Models\Contents\Exceptions\NotContentApproveException;
use Mnv\Models\Contents\Exceptions\ContentNotDeletedException;

/**
 * Class Product
 * @package Mnv\Models\Contents
 */
class Product extends AbstractContent
{

    private static $_table              = 'articles';
    private static $_table_images       = 'article_images';
    private static $_table_comments     = 'comments';
    private static $_table_variants     = 'product_variants';
    private $_fields = array('articleId', 'title', 'alias', 'sectionId', 'isFeatured', 'url', 'publishedOn', 'orderBy', 'typeContent', 'status' , 'enableGallery');


    /**
     * Получение всех статей
     *
     * @param string|null $orderBy
     * @return mixed|void
     */
    protected function listContent(?string $orderBy = 'publishedOn DESC'): void
    {
        /** фильтрация / сортировка */
        $this->sorting();

        $this->contents = DB::init()->connect()->table(self::$_table)->select($this->_fields)->orderBy($orderBy)->pagination($this->limit, $this->page)->indexKey('articleId')->getAllIndexes('array');

        $this->total();
    }

    /**
     * Получение кол-во контента
     */
    protected function total(): void
    {
        /** фильтрация / сортировка */
        $this->sorting();

        $this->total = (int)DB::init()->connect()->table(self::$_table)->count('*', 'count')->getValue();
    }

    /**
     * Получить контент
     */
    protected function getContent() : void
    {
        $this->content = DB::init()->connect()->table(self::$_table)->where('articleId',  $this->articleId)->get('array');
    }

    /**
     * Редактирование и добавление контента
     *
     * @throws NoContentException
     */
    protected function edit(): void
    {
        if (!empty($this->articleId)) {
            $this->getContent();
            if ($this->content !== null) {
                /** получать информацию о пользователе, который создал или редактировал контент */
                Managers::gettingInfoAboutUsers($this->content);
                /** получение картинок контента */
                $this->getFiles();
            } else {
                throw new NoContentException();
            }
        }

    }

    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $fileName
     * @return int|mixed|string|null
     */
    protected function checkFileName(?string $fileName)
    {
        if (empty($fileName)) {
            $maxId = DB::init()->connect()->table(self::$_table)->max('articleId')->getValue();
            return $maxId ? $maxId + 1 : 1;
        }

        if (empty($this->articleId)) {
            if ($fileName = DB::init()->connect()->table(self::$_table)->select('fileName')->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
                return $fileName;
            }
        } else if ($fileName = DB::init()->connect()->table(self::$_table)->select('fileName')->where('articleId','<>', $this->articleId)->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
            return $fileName;
        }
        return null;
    }

    /**
     * Добавление нового контента
     *
     * @param array|null $content
     * @param array|null $images
     * @param int $managerId
     * @return mixed|void
     * @throws NotInsertContentException
     */
    protected function add(?array $content, ?array $images, int $managerId)
    {
        $content['addedBy']     = $managerId;
        $content['addedOn']     = gmdate('Y-m-d H:i:s');
        $content['modifiedBy']  = $managerId;
        $content['modifiedOn']  = gmdate('Y-m-d H:i:s');
        $content['publishedOn'] = adjustTime(date('Y-m-d H:i:s', strtotime($content['publishedOn'])),  true);

        $content['orderBy']     = DB::init()->connect()->table(static::$_table)->max('orderBy')->where('typeContent', $content['typeContent'])->getValue() + 1;

        $content['votes'] = 0;

        /* insert новую запись и возвращаем ID */
        if ($this->articleId = (int)DB::init()->connect()->table(self::$_table)->insert($content)) {
            if (!empty($images['fileId'])) {
                $this->general($this->articleId, $images);
            }
            if (!empty($images['gallery'])) {
                $this->picture($this->articleId, array_keys($images['gallery']));
            }
        } else {
            throw new NotInsertContentException();
        }

    }

    /**
     * Обновление контента
     *
     * @param array|null $content
     * @param array|null $images
     * @param int $managerId
     * @return mixed|void
     * @throws NotUpdateContentException
     */

    protected function update(?array $content, ?array $images, int $managerId)
    {
        if (!empty($this->articleId)) {
            $content['modifiedBy']   = $managerId;
            $content['modifiedOn']   = gmdate('Y-m-d H:i:s');
            $content['publishedOn'] = adjustTime(date('Y-m-d H:i:s', strtotime($content['publishedOn'])),  true);

            if (DB::init()->connect()->table(self::$_table)->where('articleId',$this->articleId)->update($content)) {

                if (!empty($images['fileId'])) {
                    $this->general($this->articleId, $images);
                }
                if (!empty($images['gallery'])) {
                    $this->picture($this->articleId, array_keys($images['gallery']));
                }

                $this->getContent();
//            if (!empty($content->sectionIds)) $content->sectionIds = json_decode($content->sectionIds);
                /* получать информацию о пользователе, который создал или редактировал раздела */
                Managers::gettingInfoAboutUsers($this->content);
                /* получение картинок раздела */
                $this->getFiles();
            } else {
                throw new NotUpdateContentException();
            }

        }
    }

    /**
     * Удаление контента
     *
     * @return mixed|void
     * @throws ContentNotDeletedException
     * @throws EmptyArticleIdException
     * @throws NoContentException
     */
    protected function remove()
    {
        if ( $this->articleId !== null ) {
            $this->getContent();
            if ($this->content !== null) {
                /** удаление контента и удаление записей из базы данных прикрепленных картинок и комментариев к этому контенту */
                if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->delete()) {
                    DB::init()->connect()->table(self::$_table_comments)->where('articleId', $this->articleId)->delete();
                    DB::init()->connect()->table(self::$_table_images)->where('articleId', $this->articleId)->delete();

                    if (DB::init()->connect()->table(self::$_table_variants)->where('productId', $this->articleId)->get()) {
                        DB::init()->connect()->table(self::$_table_variants)->where('productId', $this->articleId)->delete();
                    }

                    if (DB::init()->connect()->table('product_options')->where('productId', $this->articleId)->get()) {
                        DB::init()->connect()->table('product_options')->where('productId', $this->articleId)->delete();
                    }

                    return true;
                }
                throw new ContentNotDeletedException();
            }

            throw new NoContentException();
        }

        throw new EmptyArticleIdException();
    }

    /**
     * Обновление статуса
     *
     * @return mixed|void
     * @throws EmptyArticleIdException
     * @throws NoContentException
     * @throws NotContentApproveException
     */
    protected function approve()
    {
        if ( $this->articleId !== null ) {
            $this->getContent();
            if ($this->content !== null) {
                if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->update(['status' => 'visible'])) {
                  return true;
                }
                throw new NotContentApproveException();
            }

            throw new NoContentException();
        }

        throw new EmptyArticleIdException();
    }

    /**
     * Изменение статуса
     *
     * @throws NoContentException
     * @throws EmptyArticleIdException
     * @throws NotContentApproveException
     */
    protected function status(): bool
    {
        if ($this->articleId !== null) {
            $this->getContent();
            if ($this->content !== null) {
                $contentUpdate['status'] = ($this->content['status'] === 'visible') ? 'hidden' : 'visible';
                if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->update($contentUpdate)) {
                    $this->result = array('data' => true, 'status' => $contentUpdate['status']);
                    return true;
                }

                throw new NotContentApproveException();
            }

            throw new NoContentException();
        }

        throw new EmptyArticleIdException();
    }

    /**
     * Групповые действия
     * @param $group_action
     */
    protected function group($group_action): void
    {
        if ($group_action === 'status') {
            $this->getContent();
            if ($this->content !== null) {
                $contentUpdate['status'] = ($this->content['status'] === 'visible') ? 'hidden' : 'visible';
                $this->statusContent($contentUpdate);
            }

        } else if ($group_action === 'remove') {
            $this->getContent();
            if ($this->content !== null) {
                $this->removeContent();
            }
        }
    }

    /**
     * Обновление статуса контента
     *
     * @param $contentUpdate
     * @return bool
     */
    protected function statusContent($contentUpdate): bool
    {
        if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->update($contentUpdate)) {
            return true;
        }

        return false;
    }

    /**
     * Удаление контента
     */
    protected function removeContent(): bool
    {
        /** удаление контента и удаление записей из базы данных прикрепленных картинок и комментариев к этому контенту */
        if (DB::init()->connect()->table(self::$_table)->where('articleId', $this->articleId)->delete()) {
            DB::init()->connect()->table(self::$_table_comments)->where('articleId', $this->articleId)->delete();
            DB::init()->connect()->table(self::$_table_images)->where('articleId', $this->articleId)->delete();

            if (DB::init()->connect()->table(self::$_table_variants)->where('productId', $this->articleId)->get()) {
                DB::init()->connect()->table(self::$_table_variants)->where('productId', $this->articleId)->delete();
            }

            if (DB::init()->connect()->table('product_options')->where('productId', $this->articleId)->get()) {
                DB::init()->connect()->table('product_options')->where('productId', $this->articleId)->delete();
            }

            return true;
        }

        return false;
    }

    /**
     * Удаление основной картинки
     * @param $imageId
     * @return bool
     */
    protected function deleteImage($imageId): bool
    {
        if (!empty($imageId)) {
            DB::init()->connect()->table(self::$_table_images)->where('imageId', $imageId)->delete();
            return true;
        }

        return false;
    }

    /**
     * Добавление или замена информации у картинки в галереи
     *
     * @param $imageId
     * @param $picture
     * @return array|false|int
     */
    protected function editPictureInfo($imageId, $picture)
    {

        $update = [
            'title'         => $picture['title'],
            'alias'         => $picture['alias'],
            'description'   => $picture['description'],
            'link'          => $picture['link'],
            'position'      => $picture['position']
        ];

        if (DB::init()->connect()->table(self::$_table_images)->where('imageId', $imageId)->update($update)) {
            return DB::init()->connect()->table(self::$_table_images)->where('imageId', $imageId)->get();
        }

        return false;
    }

    /**
     * Сортировка картинок в галереи
     * @param $imageIds
     * @return bool
     */
    protected function sortPictures($imageIds): bool
    {
        $order = 1;
        foreach ($imageIds as  $imageId) {
            DB::init()->connect()->table(self::$_table_images)->where('imageId','=',$imageId)->update(['orderBy' => $order]);
            $order++;
        }
        return true;
    }

    /**
     * Удаление картинки из галереи
     * @param $imageId
     * @return bool
     */
    protected function deletePicture($imageId): bool
    {
        if (DB::init()->connect()->table(self::$_table_images)->where('imageId', '=', $imageId)->delete()) {
            return true;
        }

        return false;
    }

    /**
     * Добавление основной картинки к контенту
     *
     * @param $articleId
     * @param $images
     * @return bool
     */
    protected function general($articleId, $images): bool
    {
        $imageUpload = array_filter(['articleId' => $articleId, 'fileId' => $images['fileId'], 'type' => 'general', 'orderBy' => 0]);

        if ($imageId = DB::init()->connect()->table(self::$_table_images)->select('imageId')->where('type', '=', 'general')->where('articleId', '=', $articleId)->getValue()) {
            $imageUpload['imageId'] = $imageId;
        }

        if (DB::init()->connect()->table(self::$_table_images)->replace($imageUpload)) {
            return true;
        }

        return false;
    }

    /**
     * Загрузка картинок в галерею
     *
     * @param $articleId
     * @param $images
     */
    protected function picture($articleId, $images): void
    {
        $idx = 0;
        if ($order = DB::init()->connect()->table(self::$_table_images)->max('orderBy')->where('articleId', $articleId)->getValue()) {
            $idx = $order;
        }

        foreach ($images as $fileId) {
            $idx++;
            $imageUpload = array_filter(['articleId' => $articleId, 'fileId' => $fileId, 'type' => 'gallery', 'orderBy' => $idx]);
            DB::init()->connect()->table(self::$_table_images)->replace($imageUpload);
        }
    }

}